/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * This is not the original file distributed by the Apache Software Foundation
 * It has been modified by the Hipparchus project
 */

package org.hipparchus.geometry.euclidean.threed;

import org.hipparchus.Field;
import org.hipparchus.analysis.differentiation.DSFactory;
import org.hipparchus.analysis.differentiation.DerivativeStructure;
import org.hipparchus.exception.MathIllegalArgumentException;
import org.hipparchus.exception.MathRuntimeException;
import org.hipparchus.random.Well1024a;
import org.hipparchus.util.Binary64;
import org.hipparchus.util.Binary64Field;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.Precision;
import org.junit.jupiter.api.Test;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.util.Locale;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotEquals;
import static org.junit.jupiter.api.Assertions.assertNotSame;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;

class FieldVector3DTest {

    @Test
    void testConstructors() throws MathIllegalArgumentException {
        double cosAlpha = 1 / 2.0;
        double sinAlpha = FastMath.sqrt(3) / 2.0;
        double cosDelta = FastMath.sqrt(2) / 2.0;
        double sinDelta = -FastMath.sqrt(2) / 2.0;
        DSFactory factory21 = new DSFactory(2, 1);
        DSFactory factory31 = new DSFactory(3, 1);
        DSFactory factory41 = new DSFactory(4, 1);
        FieldVector3D<DerivativeStructure> u = new FieldVector3D<>(2,
                                                                   new FieldVector3D<>(factory21.variable(0,  FastMath.PI / 3),
                                                                                       factory21.variable(1, -FastMath.PI / 4)));
        checkVector(u, 2 * cosAlpha * cosDelta, 2 * sinAlpha * cosDelta, 2 * sinDelta);
        assertEquals(-2 * sinAlpha * cosDelta, u.getX().getPartialDerivative(1, 0), 1.0e-12);
        assertEquals(+2 * cosAlpha * cosDelta, u.getY().getPartialDerivative(1, 0), 1.0e-12);
        assertEquals(0,                        u.getZ().getPartialDerivative(1, 0), 1.0e-12);
        assertEquals(-2 * cosAlpha * sinDelta, u.getX().getPartialDerivative(0, 1), 1.0e-12);
        assertEquals(-2 * sinAlpha * sinDelta, u.getY().getPartialDerivative(0, 1), 1.0e-12);
        assertEquals(2 * cosDelta,             u.getZ().getPartialDerivative(0, 1), 1.0e-12);

        checkVector(new FieldVector3D<>(2, createVector(1, 0,  0, 3)),
                    2, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 2);
        checkVector(new FieldVector3D<>(factory41.variable(3,  2.0),
                                        createVector(1, 0,  0, 4)),
                    2, 0, 0, 2, 0, 0, 1, 0, 2, 0, 0, 0, 0, 2, 0);
        checkVector(new FieldVector3D<>(factory41.variable(3,  2.0),
                                        new Vector3D(1, 0,  0)),
                    2, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0);

        checkVector(new FieldVector3D<>(2, createVector(1, 0,  0, 3),
                                       -3, createVector(0, 0, -1, 3)),
                    2, 0, 3, -1, 0, 0, 0, -1, 0, 0, 0, -1);
        checkVector(new FieldVector3D<>(factory41.variable(3,  2.0),
                                        createVector(1, 0,  0, 4),
                                        factory41.variable(3, -3.0),
                                        createVector(0, 0, -1, 4)),
                    2, 0, 3, -1, 0, 0, 1, 0, -1, 0, 0, 0, 0, -1, -1);
        checkVector(new FieldVector3D<>(factory41.variable(3,  2.0),
                                        new Vector3D(1, 0,  0),
                                        factory41.variable(3, -3.0),
                                        new Vector3D(0, 0, -1)),
                    2, 0, 3, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, -1);

        checkVector(new FieldVector3D<>(2, createVector(1, 0, 0, 3),
                                        5, createVector(0, 1, 0, 3),
                                       -3, createVector(0, 0, -1, 3)),
                    2, 5, 3, 4, 0, 0, 0, 4, 0, 0, 0, 4);
        checkVector(new FieldVector3D<>(factory41.variable(3,  2.0),
                                        createVector(1, 0,  0, 4),
                                        factory41.variable(3,  5.0),
                                        createVector(0, 1,  0, 4),
                                        factory41.variable(3, -3.0),
                                        createVector(0, 0, -1, 4)),
                    2, 5, 3, 4, 0, 0, 1, 0, 4, 0, 1, 0, 0, 4, -1);
        checkVector(new FieldVector3D<>(factory41.variable(3,  2.0),
                                        new Vector3D(1, 0,  0),
                                        factory41.variable(3,  5.0),
                                        new Vector3D(0, 1,  0),
                                        factory41.variable(3, -3.0),
                                        new Vector3D(0, 0, -1)),
                    2, 5, 3, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, -1);

        checkVector(new FieldVector3D<>(2, createVector(1, 0, 0, 3),
                                        5, createVector(0, 1, 0, 3),
                                        5, createVector(0, -1, 0, 3),
                                       -3, createVector(0, 0, -1, 3)),
                    2, 0, 3, 9, 0, 0, 0, 9, 0, 0, 0, 9);
        checkVector(new FieldVector3D<>(factory41.variable(3,  2.0),
                                        createVector(1, 0,  0, 4),
                                        factory41.variable(3,  5.0),
                                        createVector(0, 1,  0, 4),
                                        factory41.variable(3,  5.0),
                                        createVector(0, -1,  0, 4),
                                        factory41.variable(3, -3.0),
                                        createVector(0, 0, -1, 4)),
                    2, 0, 3, 9, 0, 0, 1, 0, 9, 0, 0, 0, 0, 9, -1);
        checkVector(new FieldVector3D<>(factory41.variable(3,  2.0),
                                        new Vector3D(1, 0,  0),
                                        factory41.variable(3,  5.0),
                                        new Vector3D(0, 1,  0),
                                        factory41.variable(3,  5.0),
                                        new Vector3D(0, -1,  0),
                                        factory41.variable(3, -3.0),
                                        new Vector3D(0, 0, -1)),
                    2, 0, 3, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, -1);

        checkVector(new FieldVector3D<DerivativeStructure>(new DerivativeStructure[] {
            factory31.variable(2,  2),
            factory31.variable(1,  5),
            factory31.variable(0, -3)
        }),
        2, 5, -3, 0, 0, 1, 0, 1, 0, 1, 0, 0);

    }

    @SuppressWarnings("unlikely-arg-type")
    @Test
    void testEquals() {
        DSFactory factory31 = new DSFactory(3, 1);
        FieldVector3D<DerivativeStructure> u1 = createVector(1, 2, 3, 3);
        FieldVector3D<DerivativeStructure> v  = createVector(1, 2, 3 + 10 * Precision.EPSILON, 3);
        assertEquals(u1, u1);
        assertEquals(u1, new FieldVector3D<>(factory31.variable(0, 1.0),
            factory31.variable(1, 2.0),
            factory31.variable(2, 3.0)));
        assertNotEquals(u1, new FieldVector3D<>(factory31.constant(1.0),
            factory31.variable(1, 2.0),
            factory31.variable(2, 3.0)));
        assertNotEquals(u1, new FieldVector3D<>(factory31.variable(0, 1.0),
            factory31.constant(2.0),
            factory31.variable(2, 3.0)));
        assertNotEquals(u1, new FieldVector3D<>(factory31.variable(0, 1.0),
            factory31.variable(1, 2.0),
            factory31.constant(3.0)));
        assertNotEquals(u1, v);
        assertNotEquals(u1, u1.toVector3D());
        assertEquals(createVector(0, Double.NaN, 0, 3), createVector(0, 0, Double.NaN, 3));
    }

    @Test
    void testHash() {
        assertEquals(createVector(0, Double.NaN, 0, 3).hashCode(), createVector(0, 0, Double.NaN, 3).hashCode());
        FieldVector3D<DerivativeStructure> u = createVector(1, 2, 3, 3);
        FieldVector3D<DerivativeStructure> v = createVector(1, 2, 3 + 10 * Precision.EPSILON, 3);
        assertTrue(u.hashCode() != v.hashCode());
    }

    @Test
    void testInfinite() {
        assertTrue(createVector(1, 1, Double.NEGATIVE_INFINITY, 3).isInfinite());
        assertTrue(createVector(1, Double.NEGATIVE_INFINITY, 1, 3).isInfinite());
        assertTrue(createVector(Double.NEGATIVE_INFINITY, 1, 1, 3).isInfinite());
        assertFalse(createVector(1, 1, 2, 3).isInfinite());
        assertFalse(createVector(1, Double.NaN, Double.NEGATIVE_INFINITY, 3).isInfinite());
    }

    @Test
    void testNaN() {
        assertTrue(createVector(1, 1, Double.NaN, 3).isNaN());
        assertTrue(createVector(1, Double.NaN, 1, 3).isNaN());
        assertTrue(createVector(Double.NaN, 1, 1, 3).isNaN());
        assertFalse(createVector(1, 1, 2, 3).isNaN());
        assertFalse(createVector(1, 1, Double.NEGATIVE_INFINITY, 3).isNaN());
    }

    @Test
    void testCanonical() {

        final Field<Binary64> field = Binary64Field.getInstance();

        assertEquals(0.0, FieldVector3D.getZero(field).getNorm().getReal(), 1.0e-20);
        assertEquals(0.0, FieldVector3D.angle(FieldVector3D.getPlusI(field), Vector3D.PLUS_I).getReal(),   1.0e-20);
        assertEquals(0.0, FieldVector3D.angle(FieldVector3D.getMinusI(field), Vector3D.MINUS_I).getReal(), 1.0e-20);
        assertEquals(0.0, FieldVector3D.angle(FieldVector3D.getPlusJ(field), Vector3D.PLUS_J).getReal(),   1.0e-20);
        assertEquals(0.0, FieldVector3D.angle(FieldVector3D.getMinusJ(field), Vector3D.MINUS_J).getReal(), 1.0e-20);
        assertEquals(0.0, FieldVector3D.angle(FieldVector3D.getPlusK(field), Vector3D.PLUS_K).getReal(),   1.0e-20);
        assertEquals(0.0, FieldVector3D.angle(FieldVector3D.getMinusK(field), Vector3D.MINUS_K).getReal(), 1.0e-20);
        assertTrue(FieldVector3D.getNaN(field).isNaN());
        assertTrue(FieldVector3D.getPositiveInfinity(field).isInfinite());
        assertTrue(FieldVector3D.getNegativeInfinity(field).isInfinite());

        // new instances are created each time
        assertNotSame(FieldVector3D.getZero(field),            FieldVector3D.getZero(field));
        assertNotSame(FieldVector3D.getPlusI(field),           FieldVector3D.getPlusI(field));
        assertNotSame(FieldVector3D.getMinusI(field),          FieldVector3D.getMinusI(field));
        assertNotSame(FieldVector3D.getPlusJ(field),           FieldVector3D.getPlusJ(field));
        assertNotSame(FieldVector3D.getMinusJ(field),          FieldVector3D.getMinusJ(field));
        assertNotSame(FieldVector3D.getPlusK(field),           FieldVector3D.getPlusK(field));
        assertNotSame(FieldVector3D.getMinusK(field),          FieldVector3D.getMinusK(field));
        assertNotSame(FieldVector3D.getNaN(field),             FieldVector3D.getNaN(field));
        assertNotSame(FieldVector3D.getPositiveInfinity(field),FieldVector3D.getPositiveInfinity(field));
        assertNotSame(FieldVector3D.getNegativeInfinity(field),FieldVector3D.getNegativeInfinity(field));

    }

    @Test
    void testToString() {
        assertEquals("{3; 2; 1}", createVector(3, 2, 1, 3).toString());
        NumberFormat format = new DecimalFormat("0.000", new DecimalFormatSymbols(Locale.US));
        assertEquals("{3.000; 2.000; 1.000}", createVector(3, 2, 1, 3).toString(format));
    }

    @Test
    void testWrongDimension() throws MathIllegalArgumentException {
        assertThrows(MathIllegalArgumentException.class, () -> {
            DSFactory factory31 = new DSFactory(3, 1);
            new FieldVector3D<DerivativeStructure>(new DerivativeStructure[]{
                factory31.variable(0, 2),
                factory31.variable(0, 5)
            });
        });
    }

    @Test
    void testCoordinates() {
        FieldVector3D<DerivativeStructure> v = createVector(1, 2, 3, 3);
        assertTrue(FastMath.abs(v.getX().getReal() - 1) < 1.0e-12);
        assertTrue(FastMath.abs(v.getY().getReal() - 2) < 1.0e-12);
        assertTrue(FastMath.abs(v.getZ().getReal() - 3) < 1.0e-12);
        DerivativeStructure[] coordinates = v.toArray();
        assertTrue(FastMath.abs(coordinates[0].getReal() - 1) < 1.0e-12);
        assertTrue(FastMath.abs(coordinates[1].getReal() - 2) < 1.0e-12);
        assertTrue(FastMath.abs(coordinates[2].getReal() - 3) < 1.0e-12);
    }

    @Test
    void testNorm1() {
        assertEquals( 0.0, createVector(0, 0, 0, 3).getNorm1().getReal(), 0);
        assertEquals( 6.0, createVector(1, -2, 3, 3).getNorm1().getReal(), 0);
        assertEquals( 1.0, createVector(1, -2, 3, 3).getNorm1().getPartialDerivative(1, 0, 0), 0);
        assertEquals(-1.0, createVector(1, -2, 3, 3).getNorm1().getPartialDerivative(0, 1, 0), 0);
        assertEquals( 1.0, createVector(1, -2, 3, 3).getNorm1().getPartialDerivative(0, 0, 1), 0);
    }

    @Test
    void testNorm() {
        double r = FastMath.sqrt(14);
        assertEquals(0.0, createVector(0, 0, 0, 3).getNorm().getReal(), 0);
        assertEquals(r, createVector(1, 2, 3, 3).getNorm().getReal(), 1.0e-12);
        assertEquals( 1.0 / r, createVector(1, 2, 3, 3).getNorm().getPartialDerivative(1, 0, 0), 0);
        assertEquals( 2.0 / r, createVector(1, 2, 3, 3).getNorm().getPartialDerivative(0, 1, 0), 0);
        assertEquals( 3.0 / r, createVector(1, 2, 3, 3).getNorm().getPartialDerivative(0, 0, 1), 0);
    }

    @Test
    void testNormSq() {
        assertEquals(0.0, createVector(0, 0, 0, 3).getNormSq().getReal(), 0);
        assertEquals(14, createVector(1, 2, 3, 3).getNormSq().getReal(), 1.0e-12);
        assertEquals( 2, createVector(1, 2, 3, 3).getNormSq().getPartialDerivative(1, 0, 0), 0);
        assertEquals( 4, createVector(1, 2, 3, 3).getNormSq().getPartialDerivative(0, 1, 0), 0);
        assertEquals( 6, createVector(1, 2, 3, 3).getNormSq().getPartialDerivative(0, 0, 1), 0);
    }

    @Test
    void testNormInf() {
        assertEquals( 0.0, createVector(0, 0, 0, 3).getNormInf().getReal(), 0);
        assertEquals( 3.0, createVector(1, -2, 3, 3).getNormInf().getReal(), 0);
        assertEquals( 0.0, createVector(1, -2, 3, 3).getNormInf().getPartialDerivative(1, 0, 0), 0);
        assertEquals( 0.0, createVector(1, -2, 3, 3).getNormInf().getPartialDerivative(0, 1, 0), 0);
        assertEquals( 1.0, createVector(1, -2, 3, 3).getNormInf().getPartialDerivative(0, 0, 1), 0);
        assertEquals( 3.0, createVector(2, -1, 3, 3).getNormInf().getReal(), 0);
        assertEquals( 0.0, createVector(2, -1, 3, 3).getNormInf().getPartialDerivative(1, 0, 0), 0);
        assertEquals( 0.0, createVector(2, -1, 3, 3).getNormInf().getPartialDerivative(0, 1, 0), 0);
        assertEquals( 1.0, createVector(2, -1, 3, 3).getNormInf().getPartialDerivative(0, 0, 1), 0);
        assertEquals( 3.0, createVector(1, -3, 2, 3).getNormInf().getReal(), 0);
        assertEquals( 0.0, createVector(1, -3, 2, 3).getNormInf().getPartialDerivative(1, 0, 0), 0);
        assertEquals(-1.0, createVector(1, -3, 2, 3).getNormInf().getPartialDerivative(0, 1, 0), 0);
        assertEquals( 0.0, createVector(1, -3, 2, 3).getNormInf().getPartialDerivative(0, 0, 1), 0);
        assertEquals( 3.0, createVector(2, -3, 1, 3).getNormInf().getReal(), 0);
        assertEquals( 0.0, createVector(2, -3, 1, 3).getNormInf().getPartialDerivative(1, 0, 0), 0);
        assertEquals(-1.0, createVector(2, -3, 1, 3).getNormInf().getPartialDerivative(0, 1, 0), 0);
        assertEquals( 0.0, createVector(2, -3, 1, 3).getNormInf().getPartialDerivative(0, 0, 1), 0);
        assertEquals( 3.0, createVector(3, -1, 2, 3).getNormInf().getReal(), 0);
        assertEquals( 1.0, createVector(3, -1, 2, 3).getNormInf().getPartialDerivative(1, 0, 0), 0);
        assertEquals( 0.0, createVector(3, -1, 2, 3).getNormInf().getPartialDerivative(0, 1, 0), 0);
        assertEquals( 0.0, createVector(3, -1, 2, 3).getNormInf().getPartialDerivative(0, 0, 1), 0);
        assertEquals( 3.0, createVector(3, -2, 1, 3).getNormInf().getReal(), 0);
        assertEquals( 1.0, createVector(3, -2, 1, 3).getNormInf().getPartialDerivative(1, 0, 0), 0);
        assertEquals( 0.0, createVector(3, -2, 1, 3).getNormInf().getPartialDerivative(0, 1, 0), 0);
        assertEquals( 0.0, createVector(3, -2, 1, 3).getNormInf().getPartialDerivative(0, 0, 1), 0);
    }

    @Test
    void testDistance1() {
        FieldVector3D<DerivativeStructure> v1 = createVector(1, -2, 3, 3);
        FieldVector3D<DerivativeStructure> v2 = createVector(-4, 2, 0, 3);
        assertEquals(0.0, FieldVector3D.distance1(createVector(-1, 0, 0, 3), createVector(-1, 0, 0, 3)).getReal(), 0);
        DerivativeStructure distance = FieldVector3D.distance1(v1, v2);
        assertEquals(12.0, distance.getReal(), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(0, 0, 1), 1.0e-12);
        distance = FieldVector3D.distance1(v1, new Vector3D(-4, 2, 0));
        assertEquals(12.0, distance.getReal(), 1.0e-12);
        assertEquals( 1, distance.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(-1, distance.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals( 1, distance.getPartialDerivative(0, 0, 1), 1.0e-12);
        distance = FieldVector3D.distance1(new Vector3D(-4, 2, 0), v1);
        assertEquals(12.0, distance.getReal(), 1.0e-12);
        assertEquals( 1, distance.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(-1, distance.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals( 1, distance.getPartialDerivative(0, 0, 1), 1.0e-12);
    }

    @Test
    void testDistance() {
        FieldVector3D<DerivativeStructure> v1 = createVector(1, -2, 3, 3);
        FieldVector3D<DerivativeStructure> v2 = createVector(-4, 2, 0, 3);
        assertEquals(0.0, FieldVector3D.distance(createVector(-1, 0, 0, 3), createVector(-1, 0, 0, 3)).getReal(), 0);
        DerivativeStructure distance = FieldVector3D.distance(v1, v2);
        assertEquals(FastMath.sqrt(50), distance.getReal(), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(0, 0, 1), 1.0e-12);
        distance = FieldVector3D.distance(v1, new Vector3D(-4, 2, 0));
        assertEquals(FastMath.sqrt(50), distance.getReal(), 1.0e-12);
        assertEquals( 5 / FastMath.sqrt(50), distance.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(-4 / FastMath.sqrt(50), distance.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals( 3 / FastMath.sqrt(50), distance.getPartialDerivative(0, 0, 1), 1.0e-12);
        distance = FieldVector3D.distance(new Vector3D(-4, 2, 0), v1);
        assertEquals(FastMath.sqrt(50), distance.getReal(), 1.0e-12);
        assertEquals( 5 / FastMath.sqrt(50), distance.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(-4 / FastMath.sqrt(50), distance.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals( 3 / FastMath.sqrt(50), distance.getPartialDerivative(0, 0, 1), 1.0e-12);
    }

    @Test
    void testDistanceSq() {
        FieldVector3D<DerivativeStructure> v1 = createVector(1, -2, 3, 3);
        FieldVector3D<DerivativeStructure> v2 = createVector(-4, 2, 0, 3);
        assertEquals(0.0, FieldVector3D.distanceSq(createVector(-1, 0, 0, 3), createVector(-1, 0, 0, 3)).getReal(), 0);
        DerivativeStructure distanceSq = FieldVector3D.distanceSq(v1, v2);
        assertEquals(50.0, distanceSq.getReal(), 1.0e-12);
        assertEquals(0, distanceSq.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(0, distanceSq.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals(0, distanceSq.getPartialDerivative(0, 0, 1), 1.0e-12);
        distanceSq = FieldVector3D.distanceSq(v1, new Vector3D(-4, 2, 0));
        assertEquals(50.0, distanceSq.getReal(), 1.0e-12);
        assertEquals(10, distanceSq.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(-8, distanceSq.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals( 6, distanceSq.getPartialDerivative(0, 0, 1), 1.0e-12);
        distanceSq = FieldVector3D.distanceSq(new Vector3D(-4, 2, 0), v1);
        assertEquals(50.0, distanceSq.getReal(), 1.0e-12);
        assertEquals(10, distanceSq.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(-8, distanceSq.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals( 6, distanceSq.getPartialDerivative(0, 0, 1), 1.0e-12);
  }

    @Test
    void testDistanceInf() {
        FieldVector3D<DerivativeStructure> v1 = createVector(1, -2, 3, 3);
        FieldVector3D<DerivativeStructure> v2 = createVector(-4, 2, 0, 3);
        assertEquals(0.0, FieldVector3D.distanceInf(createVector(-1, 0, 0, 3), createVector(-1, 0, 0, 3)).getReal(), 0);
        DerivativeStructure distance = FieldVector3D.distanceInf(v1, v2);
        assertEquals(5.0, distance.getReal(), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(0, 0, 1), 1.0e-12);
        distance = FieldVector3D.distanceInf(v1, new Vector3D(-4, 2, 0));
        assertEquals(5.0, distance.getReal(), 1.0e-12);
        assertEquals(1, distance.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(0, 0, 1), 1.0e-12);
        distance = FieldVector3D.distanceInf(new Vector3D(-4, 2, 0), v1);
        assertEquals(5.0, distance.getReal(), 1.0e-12);
        assertEquals(1, distance.getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals(0, distance.getPartialDerivative(0, 0, 1), 1.0e-12);
        assertEquals(v1.subtract(v2).getNormInf().getReal(), FieldVector3D.distanceInf(v1, v2).getReal(), 1.0e-12);

        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector( 1, -2, 3, 3), createVector(-4,  2, 0, 3)).getReal(),
                            1.0e-12);
        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector( 1, 3, -2, 3), createVector(-4, 0,  2, 3)).getReal(),
                            1.0e-12);
        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector(-2,  1, 3, 3), createVector( 2, -4, 0, 3)).getReal(),
                            1.0e-12);
        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector(-2, 3,  1, 3), createVector( 2, 0, -4, 3)).getReal(),
                            1.0e-12);
        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector(3, -2,  1, 3), createVector(0,  2, -4, 3)).getReal(),
                            1.0e-12);
        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector(3,  1, -2, 3), createVector(0, -4,  2, 3)).getReal(),
                            1.0e-12);

        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector( 1, -2, 3, 3), new Vector3D(-4,  2, 0)).getReal(),
                            1.0e-12);
        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector( 1, 3, -2, 3), new Vector3D(-4, 0,  2)).getReal(),
                            1.0e-12);
        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector(-2,  1, 3, 3), new Vector3D( 2, -4, 0)).getReal(),
                            1.0e-12);
        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector(-2, 3,  1, 3), new Vector3D( 2, 0, -4)).getReal(),
                            1.0e-12);
        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector(3, -2,  1, 3), new Vector3D(0,  2, -4)).getReal(),
                            1.0e-12);
        assertEquals(5.0,
                            FieldVector3D.distanceInf(createVector(3,  1, -2, 3), new Vector3D(0, -4,  2)).getReal(),
                            1.0e-12);

    }

    @Test
    void testSubtract() {
        FieldVector3D<DerivativeStructure> v1 = createVector(1, 2, 3, 3);
        FieldVector3D<DerivativeStructure> v2 = createVector(-3, -2, -1, 3);
        v1 = v1.subtract(v2);
        checkVector(v1, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0);

        checkVector(v2.subtract(v1), -7, -6, -5, 1, 0, 0, 0, 1, 0, 0, 0, 1);
        checkVector(v2.subtract(new Vector3D(4, 4, 4)), -7, -6, -5, 1, 0, 0, 0, 1, 0, 0, 0, 1);
        checkVector(v2.subtract(3, v1), -15, -14, -13, 1, 0, 0, 0, 1, 0, 0, 0, 1);
        checkVector(v2.subtract(3, new Vector3D(4, 4, 4)), -15, -14, -13, 1, 0, 0, 0, 1, 0, 0, 0, 1);
        DSFactory factory31 = new DSFactory(3, 1);
        checkVector(v2.subtract(factory31.variable(2, 3), new Vector3D(4, 4, 4)),
                    -15, -14, -13, 1, 0, -4, 0, 1, -4, 0, 0, -3);

        DSFactory factory41 = new DSFactory(4, 1);
        checkVector(createVector(1, 2, 3, 4).subtract(factory41.variable(3, 5.0),
                                                      createVector(3, -2, 1, 4)),
                    -14, 12, -2,
                     -4,  0,  0, -3,
                      0, -4,  0,  2,
                      0,  0, -4, -1);

    }

    @Test
    void testAdd() {
        FieldVector3D<DerivativeStructure> v1 = createVector(1, 2, 3, 3);
        FieldVector3D<DerivativeStructure> v2 = createVector(-3, -2, -1, 3);
        v1 = v1.add(v2);
        checkVector(v1, -2, 0, 2, 2, 0, 0, 0, 2, 0, 0, 0, 2);

        checkVector(v2.add(v1), -5, -2, 1, 3, 0, 0, 0, 3, 0, 0, 0, 3);
        checkVector(v2.add(new Vector3D(-2, 0, 2)), -5, -2, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1);
        checkVector(v2.add(3, v1), -9, -2, 5, 7, 0, 0, 0, 7, 0, 0, 0, 7);
        checkVector(v2.add(3, new Vector3D(-2, 0, 2)), -9, -2, 5, 1, 0, 0, 0, 1, 0, 0, 0, 1);
        DSFactory factory31 = new DSFactory(3, 1);
        checkVector(v2.add(factory31.variable(2, 3), new Vector3D(-2, 0, 2)),
                    -9, -2, 5, 1, 0, -2, 0, 1, 0, 0, 0, 3);

        DSFactory factory41 = new DSFactory(4, 1);
        checkVector(createVector(1, 2, 3, 4).add(factory41.variable(3, 5.0),
                                                 createVector(3, -2, 1, 4)),
                    16, -8,  8,
                     6,  0,  0,  3,
                     0,  6,  0, -2,
                     0,  0,  6,  1);

    }

    @Test
    void testScalarProduct() {
        FieldVector3D<DerivativeStructure> v = createVector(1, 2, 3, 3);
        v = v.scalarMultiply(3);
        checkVector(v, 3, 6, 9);

        checkVector(v.scalarMultiply(0.5), 1.5, 3, 4.5);
    }

    @Test
    void testVectorialProducts() {
        FieldVector3D<DerivativeStructure> v1 = createVector(2, 1, -4, 3);
        FieldVector3D<DerivativeStructure> v2 = createVector(3, 1, -1, 3);

        assertTrue(FastMath.abs(FieldVector3D.dotProduct(v1, v2).getReal() - 11) < 1.0e-12);
        assertTrue(FastMath.abs(FieldVector3D.dotProduct(v1, v2.toVector3D()).getReal() - 11) < 1.0e-12);
        assertTrue(FastMath.abs(FieldVector3D.dotProduct(v1.toVector3D(), v2).getReal() - 11) < 1.0e-12);

        FieldVector3D<DerivativeStructure> v3 = FieldVector3D.crossProduct(v1, v2);
        checkVector(v3, 3, -10, -1);
        assertTrue(FastMath.abs(FieldVector3D.dotProduct(v1, v3).getReal()) < 1.0e-12);
        assertTrue(FastMath.abs(FieldVector3D.dotProduct(v2, v3).getReal()) < 1.0e-12);

        v3 = FieldVector3D.crossProduct(v1, v2.toVector3D());
        checkVector(v3, 3, -10, -1);
        assertTrue(FastMath.abs(FieldVector3D.dotProduct(v1, v3).getReal()) < 1.0e-12);
        assertTrue(FastMath.abs(FieldVector3D.dotProduct(v2, v3).getReal()) < 1.0e-12);

        v3 = FieldVector3D.crossProduct(v1.toVector3D(), v2);
        checkVector(v3, 3, -10, -1);
        assertTrue(FastMath.abs(FieldVector3D.dotProduct(v1, v3).getReal()) < 1.0e-12);
        assertTrue(FastMath.abs(FieldVector3D.dotProduct(v2, v3).getReal()) < 1.0e-12);

    }

    @Test
    void testCrossProductCancellation() {
        FieldVector3D<DerivativeStructure> v1 = createVector(9070467121.0, 4535233560.0, 1, 3);
        FieldVector3D<DerivativeStructure> v2 = createVector(9070467123.0, 4535233561.0, 1, 3);
        checkVector(FieldVector3D.crossProduct(v1, v2), -1, 2, 1);

        double scale    = FastMath.scalb(1.0, 100);
        FieldVector3D<DerivativeStructure> big1   = new FieldVector3D<>(scale, v1);
        FieldVector3D<DerivativeStructure> small2 = new FieldVector3D<>(1 / scale, v2);
        checkVector(FieldVector3D.crossProduct(big1, small2), -1, 2, 1);

    }

    @Test
    void testAngular() {
        assertEquals(0,           createVector(1, 0, 0, 3).getAlpha().getReal(), 1.0e-10);
        assertEquals(0,           createVector(1, 0, 0, 3).getDelta().getReal(), 1.0e-10);
        assertEquals(FastMath.PI / 2, createVector(0, 1, 0, 3).getAlpha().getReal(), 1.0e-10);
        assertEquals(0,           createVector(0, 1, 0, 3).getDelta().getReal(), 1.0e-10);
        assertEquals(FastMath.PI / 2, createVector(0, 0, 1, 3).getDelta().getReal(), 1.0e-10);

        FieldVector3D<DerivativeStructure> u = createVector(-1, 1, -1, 3);
        assertEquals(3 * FastMath.PI /4, u.getAlpha().getReal(), 1.0e-10);
        assertEquals(-1.0 / FastMath.sqrt(3), u.getDelta().sin().getReal(), 1.0e-10);
    }

    @Test
    void testAngularSeparation() throws MathRuntimeException {
        FieldVector3D<DerivativeStructure> v1 = createVector(2, -1, 4, 3);

        FieldVector3D<DerivativeStructure>  k = v1.normalize();
        FieldVector3D<DerivativeStructure>  i = k.orthogonal();
        FieldVector3D<DerivativeStructure> v2 = k.scalarMultiply(FastMath.cos(1.2)).add(i.scalarMultiply(FastMath.sin(1.2)));

        assertTrue(FastMath.abs(FieldVector3D.angle(v1, v2).getReal() - 1.2) < 1.0e-12);
        assertTrue(FastMath.abs(FieldVector3D.angle(v1, v2.toVector3D()).getReal() - 1.2) < 1.0e-12);
        assertTrue(FastMath.abs(FieldVector3D.angle(v1.toVector3D(), v2).getReal() - 1.2) < 1.0e-12);

        try {
            FieldVector3D.angle(v1, Vector3D.ZERO);
            fail("an exception should have been thrown");
        } catch (MathRuntimeException mae) {
            // expected
        }
        assertEquals(0.0, FieldVector3D.angle(v1, v1.toVector3D()).getReal(), 1.0e-15);
        assertEquals(FastMath.PI, FieldVector3D.angle(v1, v1.negate().toVector3D()).getReal(), 1.0e-15);

    }

    @Test
    void testNormalize() throws MathRuntimeException {
        assertEquals(1.0, createVector(5, -4, 2, 3).normalize().getNorm().getReal(), 1.0e-12);
        try {
            createVector(0, 0, 0, 3).normalize();
            fail("an exception should have been thrown");
        } catch (MathRuntimeException ae) {
            // expected behavior
        }
    }

    @Test
    void testNegate() {
        checkVector(createVector(0.1, 2.5, 1.3, 3).negate(),
                    -0.1, -2.5, -1.3, -1, 0, 0, 0, -1, 0, 0, 0, -1);
    }

    @Test
    void testOrthogonal() throws MathRuntimeException {
        FieldVector3D<DerivativeStructure> v1 = createVector(0.1, 2.5, 1.3, 3);
        assertEquals(0.0, FieldVector3D.dotProduct(v1, v1.orthogonal()).getReal(), 1.0e-12);
        FieldVector3D<DerivativeStructure> v2 = createVector(2.3, -0.003, 7.6, 3);
        assertEquals(0.0, FieldVector3D.dotProduct(v2, v2.orthogonal()).getReal(), 1.0e-12);
        FieldVector3D<DerivativeStructure> v3 = createVector(-1.7, 1.4, 0.2, 3);
        assertEquals(0.0, FieldVector3D.dotProduct(v3, v3.orthogonal()).getReal(), 1.0e-12);
        FieldVector3D<DerivativeStructure> v4 = createVector(4.2, 0.1, -1.8, 3);
        assertEquals(0.0, FieldVector3D.dotProduct(v4, v4.orthogonal()).getReal(), 1.0e-12);
        try {
            createVector(0, 0, 0, 3).orthogonal();
            fail("an exception should have been thrown");
        } catch (MathRuntimeException ae) {
            // expected behavior
        }
    }

    @Test
    void testAngle() throws MathRuntimeException {
        assertEquals(0.22572612855273393616,
                            FieldVector3D.angle(createVector(1, 2, 3, 3), createVector(4, 5, 6, 3)).getReal(),
                            1.0e-12);
        assertEquals(7.98595620686106654517199e-8,
                            FieldVector3D.angle(createVector(1, 2, 3, 3), createVector(2, 4, 6.000001, 3)).getReal(),
                            1.0e-12);
        assertEquals(3.14159257373023116985197793156,
                            FieldVector3D.angle(createVector(1, 2, 3, 3), createVector(-2, -4, -6.000001, 3)).getReal(),
                            1.0e-12);
        try {
            FieldVector3D.angle(createVector(0, 0, 0, 3), createVector(1, 0, 0, 3));
            fail("an exception should have been thrown");
        } catch (MathRuntimeException ae) {
            // expected behavior
        }
    }

    @Test
    void testAccurateDotProduct() {
        // the following two vectors are nearly but not exactly orthogonal
        // naive dot product (i.e. computing u1.x * u2.x + u1.y * u2.y + u1.z * u2.z
        // leads to a result of 0.0, instead of the correct -1.855129...
        FieldVector3D<DerivativeStructure> u1 = createVector(-1321008684645961.0 /  268435456.0,
                                   -5774608829631843.0 /  268435456.0,
                                   -7645843051051357.0 / 8589934592.0, 3);
        FieldVector3D<DerivativeStructure> u2 = createVector(-5712344449280879.0 /    2097152.0,
                                   -4550117129121957.0 /    2097152.0,
                                    8846951984510141.0 /     131072.0, 3);
        DerivativeStructure sNaive = u1.getX().multiply(u2.getX()).add(u1.getY().multiply(u2.getY())).add(u1.getZ().multiply(u2.getZ()));
        DerivativeStructure sAccurate = FieldVector3D.dotProduct(u1, u2);
        assertEquals(0.0, sNaive.getReal(), 1.0e-30);
        assertEquals(-2088690039198397.0 / 1125899906842624.0, sAccurate.getReal(), 1.0e-15);
    }

    @Test
    void testDotProduct() {
        // we compare accurate versus naive dot product implementations
        // on regular vectors (i.e. not extreme cases like in the previous test)
        Well1024a random = new Well1024a(553267312521321234l);
        for (int i = 0; i < 10000; ++i) {
            double ux = 10000 * random.nextDouble();
            double uy = 10000 * random.nextDouble();
            double uz = 10000 * random.nextDouble();
            double vx = 10000 * random.nextDouble();
            double vy = 10000 * random.nextDouble();
            double vz = 10000 * random.nextDouble();
            double sNaive = ux * vx + uy * vy + uz * vz;

            FieldVector3D<DerivativeStructure> uds = createVector(ux, uy, uz, 3);
            FieldVector3D<DerivativeStructure> vds = createVector(vx, vy, vz, 3);
            Vector3D v = new Vector3D(vx, vy, vz);

            DerivativeStructure sAccurate = FieldVector3D.dotProduct(uds, vds);
            assertEquals(sNaive, sAccurate.getReal(), 2.5e-16 * sNaive);
            assertEquals(ux + vx, sAccurate.getPartialDerivative(1, 0, 0), 2.5e-16 * sNaive);
            assertEquals(uy + vy, sAccurate.getPartialDerivative(0, 1, 0), 2.5e-16 * sNaive);
            assertEquals(uz + vz, sAccurate.getPartialDerivative(0, 0, 1), 2.5e-16 * sNaive);

            sAccurate = FieldVector3D.dotProduct(uds, v);
            assertEquals(sNaive, sAccurate.getReal(), 2.5e-16 * sNaive);
            assertEquals(vx, sAccurate.getPartialDerivative(1, 0, 0), 2.5e-16 * sNaive);
            assertEquals(vy, sAccurate.getPartialDerivative(0, 1, 0), 2.5e-16 * sNaive);
            assertEquals(vz, sAccurate.getPartialDerivative(0, 0, 1), 2.5e-16 * sNaive);

        }
    }

    @Test
    void testAccurateCrossProduct() {
        // the vectors u1 and u2 are nearly but not exactly anti-parallel
        // (7.31e-16 degrees from 180 degrees) naive cross product (i.e.
        // computing u1.x * u2.x + u1.y * u2.y + u1.z * u2.z
        // leads to a result of   [0.0009765, -0.0001220, -0.0039062],
        // instead of the correct [0.0006913, -0.0001254, -0.0007909]
        final FieldVector3D<DerivativeStructure> u1 = createVector(-1321008684645961.0 /   268435456.0,
                                                                   -5774608829631843.0 /   268435456.0,
                                                                   -7645843051051357.0 /  8589934592.0, 3);
        final FieldVector3D<DerivativeStructure> u2 = createVector( 1796571811118507.0 /  2147483648.0,
                                                                    7853468008299307.0 /  2147483648.0,
                                                                    2599586637357461.0 / 17179869184.0, 3);
        final FieldVector3D<DerivativeStructure> u3 = createVector(12753243807587107.0 / 18446744073709551616.0,
                                                                   -2313766922703915.0 / 18446744073709551616.0,
                                                                    -227970081415313.0 /   288230376151711744.0, 3);
        FieldVector3D<DerivativeStructure> cNaive = new FieldVector3D<DerivativeStructure>(u1.getY().multiply(u2.getZ()).subtract(u1.getZ().multiply(u2.getY())),
                                       u1.getZ().multiply(u2.getX()).subtract(u1.getX().multiply(u2.getZ())),
                                       u1.getX().multiply(u2.getY()).subtract(u1.getY().multiply(u2.getX())));
        FieldVector3D<DerivativeStructure> cAccurate = FieldVector3D.crossProduct(u1, u2);
        assertTrue(FieldVector3D.distance(u3, cNaive).getReal() > 2.9 * u3.getNorm().getReal());
        assertEquals(0.0, FieldVector3D.distance(u3, cAccurate).getReal(), 1.0e-30 * cAccurate.getNorm().getReal());
    }

    @Test
    void testCrossProduct() {
        // we compare accurate versus naive cross product implementations
        // on regular vectors (i.e. not extreme cases like in the previous test)
        Well1024a random = new Well1024a(885362227452043214l);
        for (int i = 0; i < 10000; ++i) {
            double ux = random.nextDouble();
            double uy = random.nextDouble();
            double uz = random.nextDouble();
            double vx = random.nextDouble();
            double vy = random.nextDouble();
            double vz = random.nextDouble();
            Vector3D cNaive = new Vector3D(uy * vz - uz * vy, uz * vx - ux * vz, ux * vy - uy * vx);

            FieldVector3D<DerivativeStructure> uds = createVector(ux, uy, uz, 3);
            FieldVector3D<DerivativeStructure> vds = createVector(vx, vy, vz, 3);
            Vector3D v = new Vector3D(vx, vy, vz);

            checkVector(FieldVector3D.crossProduct(uds, vds),
                        cNaive.getX(), cNaive.getY(), cNaive.getZ(),
                        0, vz - uz, uy - vy,
                        uz - vz, 0, vx - ux,
                        vy - uy, ux - vx, 0);

            checkVector(FieldVector3D.crossProduct(uds, v),
                        cNaive.getX(), cNaive.getY(), cNaive.getZ(),
                          0,  vz, -vy,
                        -vz,   0,  vx,
                         vy, -vx,   0);

        }
    }

    @Test
    void testArithmeticalBlending(){
        // Given
        final FieldVector3D<Binary64> vector1 = new FieldVector3D<>(new Binary64(1), new Binary64(2), new Binary64(3));
        final FieldVector3D<Binary64> vector2 = new FieldVector3D<>(new Binary64(2), new Binary64(4), new Binary64(9));

        final Binary64 blendingValue = new Binary64(0.65);

        // When
        final FieldVector3D<Binary64> blendedVector = vector1.blendArithmeticallyWith(vector2, blendingValue);

        // Then
        assertEquals(1.65, blendedVector.getX().getReal(), 1.0e-15);
        assertEquals(3.3 , blendedVector.getY().getReal(), 1.0e-15);
        assertEquals(6.9 , blendedVector.getZ().getReal(), 1.0e-15);
    }

    private FieldVector3D<DerivativeStructure> createVector(double x, double y, double z, int params) {
        DSFactory factory = new DSFactory(params, 1);
        return new FieldVector3D<>(factory.variable(0, x),
                                   factory.variable(1, y),
                                   factory.variable(2, z));
    }

    private void checkVector(FieldVector3D<DerivativeStructure> v, double x, double y, double z) {
        assertEquals(x, v.getX().getReal(), 1.0e-12);
        assertEquals(y, v.getY().getReal(), 1.0e-12);
        assertEquals(z, v.getZ().getReal(), 1.0e-12);
    }

    private void checkVector(FieldVector3D<DerivativeStructure> v, double x, double y, double z,
                             double dxdx, double dxdy, double dxdz,
                             double dydx, double dydy, double dydz,
                             double dzdx, double dzdy, double dzdz) {
        assertEquals(x, v.getX().getReal(), 1.0e-12);
        assertEquals(y, v.getY().getReal(), 1.0e-12);
        assertEquals(z, v.getZ().getReal(), 1.0e-12);
        assertEquals(dxdx, v.getX().getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(dxdy, v.getX().getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals(dxdz, v.getX().getPartialDerivative(0, 0, 1), 1.0e-12);
        assertEquals(dydx, v.getY().getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(dydy, v.getY().getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals(dydz, v.getY().getPartialDerivative(0, 0, 1), 1.0e-12);
        assertEquals(dzdx, v.getZ().getPartialDerivative(1, 0, 0), 1.0e-12);
        assertEquals(dzdy, v.getZ().getPartialDerivative(0, 1, 0), 1.0e-12);
        assertEquals(dzdz, v.getZ().getPartialDerivative(0, 0, 1), 1.0e-12);
    }

    private void checkVector(FieldVector3D<DerivativeStructure> v, double x, double y, double z,
                             double dxdx, double dxdy, double dxdz, double dxdt,
                             double dydx, double dydy, double dydz, double dydt,
                             double dzdx, double dzdy, double dzdz, double dzdt) {
        assertEquals(x, v.getX().getReal(), 1.0e-12);
        assertEquals(y, v.getY().getReal(), 1.0e-12);
        assertEquals(z, v.getZ().getReal(), 1.0e-12);
        assertEquals(dxdx, v.getX().getPartialDerivative(1, 0, 0, 0), 1.0e-12);
        assertEquals(dxdy, v.getX().getPartialDerivative(0, 1, 0, 0), 1.0e-12);
        assertEquals(dxdz, v.getX().getPartialDerivative(0, 0, 1, 0), 1.0e-12);
        assertEquals(dxdt, v.getX().getPartialDerivative(0, 0, 0, 1), 1.0e-12);
        assertEquals(dydx, v.getY().getPartialDerivative(1, 0, 0, 0), 1.0e-12);
        assertEquals(dydy, v.getY().getPartialDerivative(0, 1, 0, 0), 1.0e-12);
        assertEquals(dydz, v.getY().getPartialDerivative(0, 0, 1, 0), 1.0e-12);
        assertEquals(dydt, v.getY().getPartialDerivative(0, 0, 0, 1), 1.0e-12);
        assertEquals(dzdx, v.getZ().getPartialDerivative(1, 0, 0, 0), 1.0e-12);
        assertEquals(dzdy, v.getZ().getPartialDerivative(0, 1, 0, 0), 1.0e-12);
        assertEquals(dzdz, v.getZ().getPartialDerivative(0, 0, 1, 0), 1.0e-12);
        assertEquals(dzdt, v.getZ().getPartialDerivative(0, 0, 0, 1), 1.0e-12);
    }

}
